<?php

class AdminController
{
    private Db $db;
    private ?JsonStore $js;
    private array $config;

    public function __construct(Db $db, ?JsonStore $js, array $config)
    {
        $this->db = $db;
        $this->js = $js;
        $this->config = $config;
    }

    public function dashboard(): void
    {
        require_admin();

        $stats = $this->stats();
        $orders = $this->recentOrders(50);

        View::render('admin/dashboard', [
            'title' => 'ادمین',
            'stats' => $stats,
            'orders' => $orders,
        ]);
    }

    public function services(): void
    {
        require_admin();
        [$cats, $sv] = $this->catalog();
        View::render('admin/services', [
            'title' => 'مدیریت سرویس‌ها',
            'categories' => $cats,
            'services' => $sv,
        ]);
    }

    public function serviceCreate(): void
    {
        require_admin();
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $title = trim((string)($_POST['title'] ?? ''));
        $desc = trim((string)($_POST['description'] ?? ''));
        $categoryId = (int)($_POST['category_id'] ?? 0);
        $min = (int)($_POST['min_qty'] ?? 0);
        $max = (int)($_POST['max_qty'] ?? 0);
        $price = (int)($_POST['unit_price'] ?? 0);
        $active = isset($_POST['active']) ? 1 : 0;

        if ($title === '' || $desc === '' || $categoryId <= 0 || $min <= 0 || $max <= 0 || $price <= 0) {
            session_flash_set('اطلاعات سرویس کامل نیست.');
            redirect(app_url('/admin/services'));
        }

        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('INSERT INTO services (category_id,title,description,min_qty,max_qty,unit_price,active) VALUES (?,?,?,?,?,?,?)');
            $st->execute([$categoryId,$title,$desc,$min,$max,$price,$active]);
        } else {
            $sv = $this->js->read('services');
            $sv[] = [
                'id'=>$this->js->nextId($sv),
                'category_id'=>$categoryId,
                'title'=>$title,
                'description'=>$desc,
                'min_qty'=>$min,
                'max_qty'=>$max,
                'unit_price'=>$price,
                'active'=>$active,
            ];
            $this->js->write('services',$sv);
        }

        session_flash_set('سرویس ثبت شد.');
        redirect(app_url('/admin/services'));
    }

    public function serviceToggle(): void
    {
        require_admin();
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $id = (int)($_POST['id'] ?? 0);
        if ($id <= 0) redirect(app_url('/admin/services'));

        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $pdo->prepare('UPDATE services SET active = CASE WHEN active=1 THEN 0 ELSE 1 END WHERE id=?')->execute([$id]);
        } else {
            $sv = $this->js->read('services');
            foreach ($sv as &$s) {
                if ((int)($s['id'] ?? 0) === $id) {
                    $s['active'] = ((int)($s['active'] ?? 0) === 1) ? 0 : 1;
                    break;
                }
            }
            $this->js->write('services',$sv);
        }

        session_flash_set('وضعیت سرویس تغییر کرد.');
        redirect(app_url('/admin/services'));
    }

    public function orderStatus(): void
    {
        require_admin();
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $id = (int)($_POST['id'] ?? 0);
        $status = (string)($_POST['status'] ?? 'pending');

        $allowed = ['pending','processing','completed','canceled'];
        if ($id <= 0 || !in_array($status, $allowed, true)) {
            session_flash_set('درخواست نامعتبر است.');
            redirect(app_url('/admin'));
        }

        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $pdo->prepare('UPDATE orders SET status=? WHERE id=?')->execute([$status,$id]);
        } else {
            $orders = $this->js->read('orders');
            foreach ($orders as &$o) {
                if ((int)($o['id'] ?? 0) === $id) {
                    $o['status'] = $status;
                    break;
                }
            }
            $this->js->write('orders',$orders);
        }

        session_flash_set('وضعیت سفارش بروزرسانی شد.');
        redirect(app_url('/admin'));
    }

    private function catalog(): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $cats = $pdo->query('SELECT * FROM categories ORDER BY id ASC')->fetchAll(PDO::FETCH_ASSOC);
            $sv = $pdo->query('SELECT * FROM services ORDER BY id ASC')->fetchAll(PDO::FETCH_ASSOC);
            return [$cats, $sv];
        }
        return [$this->js->read('categories'), $this->js->read('services')];
    }

    private function stats(): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            return [
                'users' => (int)$pdo->query('SELECT COUNT(*) FROM users')->fetchColumn(),
                'orders' => (int)$pdo->query('SELECT COUNT(*) FROM orders')->fetchColumn(),
                'revenue' => (int)$pdo->query('SELECT COALESCE(SUM(total_price),0) FROM orders')->fetchColumn(),
                'wallet' => (int)$pdo->query("SELECT COALESCE(SUM(CASE WHEN type='credit' THEN amount ELSE -amount END),0) FROM wallet_ledger")->fetchColumn(),
            ];
        }

        $users = $this->js->read('users');
        $orders = $this->js->read('orders');
        $ledger = $this->js->read('wallet_ledger');

        $wallet = 0;
        foreach ($ledger as $l) {
            $t = (string)($l['type'] ?? '');
            $a = (int)($l['amount'] ?? 0);
            $wallet += ($t === 'credit') ? $a : -$a;
        }

        $revenue = 0;
        foreach ($orders as $o) $revenue += (int)($o['total_price'] ?? 0);

        return [
            'users' => count($users),
            'orders' => count($orders),
            'revenue' => $revenue,
            'wallet' => $wallet,
        ];
    }

    private function recentOrders(int $limit): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('SELECT o.*, u.phone, s.title as service_title FROM orders o
                JOIN users u ON u.id=o.user_id
                JOIN services s ON s.id=o.service_id
                ORDER BY o.id DESC LIMIT ?');
            $st->bindValue(1, $limit, PDO::PARAM_INT);
            $st->execute();
            return $st->fetchAll(PDO::FETCH_ASSOC);
        }

        $orders = $this->js->read('orders');
        $users = $this->js->read('users');
        $services = $this->js->read('services');
        $umap = [];
        foreach ($users as $u) $umap[(int)$u['id']] = $u;
        $smap = [];
        foreach ($services as $s) $smap[(int)$s['id']] = $s;

        usort($orders, fn($a,$b)=> (int)($b['id']??0) <=> (int)($a['id']??0));
        $out = [];
        foreach (array_slice($orders, 0, $limit) as $o) {
            $o['phone'] = $umap[(int)($o['user_id'] ?? 0)]['phone'] ?? '';
            $o['service_title'] = $smap[(int)($o['service_id'] ?? 0)]['title'] ?? '';
            $out[] = $o;
        }
        return $out;
    }
}
