<?php

class AuthController
{
    private Db $db;
    private ?JsonStore $js;
    private array $config;

    public function __construct(Db $db, ?JsonStore $js, array $config)
    {
        $this->db = $db;
        $this->js = $js;
        $this->config = $config;
    }

    public function showLogin(): void
    {
        View::render('auth/login', ['title' => 'ورود']);
    }

    public function sendOtp(): void
    {
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $phone = trim((string)($_POST['phone'] ?? ''));
        if ($phone === '' || !preg_match('/^\d{10,14}$/', $phone)) {
            session_flash_set('شماره موبایل معتبر نیست.');
            redirect(app_url('/login'));
        }

        // rate limit: store last send time in session per phone (simple)
        $rl = (int)($this->config['otp']['rate_limit_seconds'] ?? 60);
        $last = (int)($_SESSION['_otp_last_'.$phone] ?? 0);
        if ($last && (time() - $last) < $rl) {
            session_flash_set('ارسال کد زیاد انجام شده. کمی بعد تلاش کنید.');
            redirect(app_url('/login'));
        }

        $code = (string)random_int(1000, 9999);
        $_SESSION['_otp_last_'.$phone] = time();

        // Persist OTP
        $ttl = (int)($this->config['otp']['otp_ttl_seconds'] ?? 300);
        $exp = time() + $ttl;

        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $stmt = $pdo->prepare('INSERT INTO otp_codes (phone, code, expires_at, created_at) VALUES (?, ?, ?, ?)');
            $stmt->execute([$phone, $code, $exp, time()]);
        } else {
            $otp = $this->js->read('otp_codes');
            $otp[] = ['id'=>$this->js->nextId($otp),'phone'=>$phone,'code'=>$code,'expires_at'=>$exp,'created_at'=>time()];
            $this->js->write('otp_codes', $otp);
        }

        $provider = new KavenegarOtp($this->config);
        $ok = $provider->send($phone, $code);

        // If real provider fails, do not pretend.
        if (!$ok) {
            session_flash_set('ارسال پیامک ناموفق بود (پیکربندی را بررسی کنید).');
            redirect(app_url('/login'));
        }

        // For demo: show a hint if debug
        if (!empty($this->config['app']['debug'])) {
            session_flash_set('کد ارسال شد (دمو/دیباگ). اگر پیامک نیامد، لاگ را ببینید یا کد دمو 1234 را وارد کنید.');
        } else {
            session_flash_set('کد ارسال شد.');
        }

        redirect(app_url('/login', ['phone'=>$phone]));
    }

    public function verifyLogin(): void
    {
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $phone = trim((string)($_POST['phone'] ?? ''));
        $otp = trim((string)($_POST['otp'] ?? ''));

        if ($phone === '' || $otp === '') {
            session_flash_set('شماره موبایل و کد الزامی است.');
            redirect(app_url('/login'));
        }

        $demo = (string)($this->config['otp']['demo_code'] ?? '1234');
        $valid = false;

        if ($otp === $demo) {
            $valid = true;
        } else {
            $valid = $this->verifyOtpFromStore($phone, $otp);
        }

        if (!$valid) {
            session_flash_set('کد وارد شده صحیح نیست یا منقضی شده.');
            redirect(app_url('/login', ['phone'=>$phone]));
        }

        // Role logic
        $prefix = (string)($this->config['security']['admin_phone_prefix'] ?? '00');
        $role = str_starts_with($phone, $prefix) ? 'admin' : 'user';

        $user = User::findOrCreateByPhone($this->db, $this->js, $phone, $role);

        // ensure role persisted (session)
        $user['role'] = $role;

        $_SESSION['user'] = $user;

        redirect(app_url('/dashboard'));
    }

    private function verifyOtpFromStore(string $phone, string $code): bool
    {
        $now = time();

        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $stmt = $pdo->prepare('SELECT * FROM otp_codes WHERE phone=? AND code=? ORDER BY id DESC LIMIT 1');
            $stmt->execute([$phone, $code]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if (!$row) return false;
            if ((int)$row['expires_at'] < $now) return false;

            // consume
            $del = $pdo->prepare('DELETE FROM otp_codes WHERE id=?');
            $del->execute([(int)$row['id']]);
            return true;
        }

        $otp = $this->js->read('otp_codes');
        $foundIndex = null;
        for ($i = count($otp)-1; $i >= 0; $i--) {
            $row = $otp[$i];
            if (($row['phone'] ?? '') === $phone && ($row['code'] ?? '') === $code) {
                if ((int)($row['expires_at'] ?? 0) < $now) return false;
                $foundIndex = $i;
                break;
            }
        }
        if ($foundIndex === null) return false;

        array_splice($otp, $foundIndex, 1);
        $this->js->write('otp_codes', $otp);
        return true;
    }

    public function logout(): void
    {
        session_destroy();
        redirect(app_url('/login'));
    }
}
