<?php

class UserController
{
    private Db $db;
    private ?JsonStore $js;
    private array $config;

    public function __construct(Db $db, ?JsonStore $js, array $config)
    {
        $this->db = $db;
        $this->js = $js;
        $this->config = $config;
    }

    public function dashboard(): void
    {
        require_login();
        $u = current_user();
        $orders = $this->getOrdersByUser((int)$u['id']);
        $wallet = $this->walletBalance((int)$u['id']);
        View::render('user/dashboard', [
            'title' => 'داشبورد',
            'user' => $u,
            'orders' => $orders,
            'wallet' => $wallet,
        ]);
    }

    public function services(): void
    {
        require_login();
        [$categories, $services] = $this->getCatalog();
        View::render('user/services', [
            'title' => 'سرویس‌ها',
            'categories' => $categories,
            'services' => $services,
        ]);
    }

    public function service(): void
    {
        require_login();
        $id = (int)($_GET['id'] ?? 0);
        $svc = $this->findService($id);
        View::render('user/service', [
            'title' => 'جزئیات سرویس',
            'service' => $svc,
        ]);
    }

    public function createOrder(): void
    {
        require_login();
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $u = current_user();
        $serviceId = (int)($_POST['service_id'] ?? 0);
        $link = trim((string)($_POST['link'] ?? ''));
        $qty = (int)($_POST['quantity'] ?? 0);

        $svc = $this->findService($serviceId);
        if (!$svc || (int)($svc['active'] ?? 0) !== 1) {
            session_flash_set('سرویس نامعتبر است.');
            redirect(app_url('/services'));
        }
        $min = (int)($svc['min_qty'] ?? 0);
        $max = (int)($svc['max_qty'] ?? 0);
        if ($link === '' || $qty < $min || $qty > $max) {
            session_flash_set('اطلاعات سفارش کامل نیست.');
            redirect(app_url('/service', ['id'=>$serviceId]));
        }

        $total = (int)($svc['unit_price'] ?? 0) * $qty;

        // For now: always pending, pay with wallet if enough, else direct payment.
        $useWallet = isset($_POST['use_wallet']) && $_POST['use_wallet'] === '1';
        if ($useWallet) {
            $bal = $this->walletBalance((int)$u['id']);
            if ($bal < $total) {
                session_flash_set('موجودی کیف پول کافی نیست.');
                redirect(app_url('/wallet'));
            }
            $this->walletAdd((int)$u['id'], 'debit', $total, 'ORDER', json_encode(['service_id'=>$serviceId], JSON_UNESCAPED_UNICODE));
        }

        $this->createOrderRow((int)$u['id'], $serviceId, $link, $qty, $total, $useWallet ? 'processing' : 'pending');

        session_flash_set('سفارش ثبت شد.');
        redirect(app_url('/orders'));
    }

    public function orders(): void
    {
        require_login();
        $u = current_user();
        $orders = $this->getOrdersByUser((int)$u['id']);
        View::render('user/orders', [
            'title' => 'سفارشات',
            'orders' => $orders,
        ]);
    }

    public function wallet(): void
    {
        require_login();
        $u = current_user();
        $bal = $this->walletBalance((int)$u['id']);
        $ledger = $this->walletLedger((int)$u['id']);
        View::render('user/wallet', [
            'title' => 'کیف پول',
            'wallet' => $bal,
            'ledger' => $ledger,
        ]);
    }

    public function startPayment(): void
    {
        require_login();
        $key = (string)($this->config['csrf']['key'] ?? '_csrf');
        csrf_verify($key);

        $u = current_user();
        $amount = (int)($_POST['amount'] ?? 0);
        $gateway = (string)($_POST['gateway'] ?? '');

        if ($amount <= 0) {
            session_flash_set('مبلغ نامعتبر است.');
            redirect(app_url('/wallet'));
        }

        // Store payment as pending
        $paymentId = $this->createPayment((int)$u['id'], $gateway, $amount);

        // If gateway not configured, mark as failed (don't lie)
        if ($gateway === 'zibal') {
            $gw = new ZibalGateway($this->config);
            if (!$gw->isConfigured()) {
                $this->updatePaymentStatus($paymentId, 'failed', null, null, ['error'=>'not configured']);
                session_flash_set('Zibal پیکربندی نشده است.');
                redirect(app_url('/wallet'));
            }
            $res = $gw->request($amount, 'Charge wallet', $u['phone'], (string)$paymentId);
            $this->updatePaymentStatus($paymentId, $res['ok'] ? 'pending' : 'failed', $res['authority'] ?? null, null, $res);
            if (!empty($res['pay_url'])) redirect($res['pay_url']);
            session_flash_set('درخواست پرداخت ناموفق بود: ' . ($res['error'] ?? ''));
            redirect(app_url('/wallet'));
        }

        if ($gateway === 'bitpay') {
            $gw = new BitPayGateway($this->config);
            if (!$gw->isConfigured()) {
                $this->updatePaymentStatus($paymentId, 'failed', null, null, ['error'=>'not configured']);
                session_flash_set('BitPay پیکربندی نشده است.');
                redirect(app_url('/wallet'));
            }
            $res = $gw->request($amount, 'Charge wallet', (string)$paymentId);
            $this->updatePaymentStatus($paymentId, $res['ok'] ? 'pending' : 'failed', $res['authority'] ?? null, null, $res);
            if (!empty($res['pay_url'])) redirect($res['pay_url']);
            session_flash_set('درخواست پرداخت ناموفق بود: ' . ($res['error'] ?? ''));
            redirect(app_url('/wallet'));
        }

        session_flash_set('درگاه نامعتبر است.');
        redirect(app_url('/wallet'));
    }

    public function paymentZibalCallback(): void
    {
        // TODO: implement after Zibal request/verify is implemented
        session_flash_set('Callback زيبال هنوز پیاده‌سازی نشده.');
        redirect(app_url('/wallet'));
    }

    public function paymentBitpayCallback(): void
    {
        // TODO: implement after BitPay request/verify is implemented
        session_flash_set('Callback بیت‌پی هنوز پیاده‌سازی نشده.');
        redirect(app_url('/wallet'));
    }

    // -------- data helpers --------

    private function getCatalog(): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $cats = $pdo->query('SELECT * FROM categories ORDER BY id ASC')->fetchAll(PDO::FETCH_ASSOC);
            $sv = $pdo->query('SELECT * FROM services ORDER BY id ASC')->fetchAll(PDO::FETCH_ASSOC);
            return [$cats, $sv];
        }
        return [$this->js->read('categories'), $this->js->read('services')];
    }

    private function findService(int $id): ?array
    {
        if ($id <= 0) return null;
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('SELECT * FROM services WHERE id=?');
            $st->execute([$id]);
            $svc = $st->fetch(PDO::FETCH_ASSOC);
            return $svc ?: null;
        }
        foreach ($this->js->read('services') as $svc) {
            if ((int)($svc['id'] ?? 0) === $id) return $svc;
        }
        return null;
    }

    private function createOrderRow(int $userId, int $serviceId, string $link, int $qty, int $total, string $status): void
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('INSERT INTO orders (user_id, service_id, link, quantity, status, total_price, created_at) VALUES (?,?,?,?,?,?,?)');
            $st->execute([$userId, $serviceId, $link, $qty, $status, $total, now()]);
            return;
        }
        $orders = $this->js->read('orders');
        $orders[] = [
            'id'=>$this->js->nextId($orders),
            'user_id'=>$userId,
            'service_id'=>$serviceId,
            'link'=>$link,
            'quantity'=>$qty,
            'status'=>$status,
            'total_price'=>$total,
            'created_at'=>now(),
        ];
        $this->js->write('orders', $orders);
    }

    private function getOrdersByUser(int $userId): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('SELECT o.*, s.title as service_title FROM orders o JOIN services s ON s.id=o.service_id WHERE o.user_id=? ORDER BY o.id DESC');
            $st->execute([$userId]);
            return $st->fetchAll(PDO::FETCH_ASSOC);
        }
        $orders = [];
        $services = $this->js->read('services');
        $map = [];
        foreach ($services as $s) $map[(int)$s['id']] = $s;
        foreach ($this->js->read('orders') as $o) {
            if ((int)($o['user_id'] ?? 0) === $userId) {
                $sid = (int)($o['service_id'] ?? 0);
                $o['service_title'] = $map[$sid]['title'] ?? ('#'.$sid);
                $orders[] = $o;
            }
        }
        usort($orders, fn($a,$b)=> (int)($b['id']??0) <=> (int)($a['id']??0));
        return $orders;
    }

    private function walletBalance(int $userId): int
    {
        $sum = 0;
        foreach ($this->walletLedger($userId) as $l) {
            $type = (string)($l['type'] ?? '');
            $amt = (int)($l['amount'] ?? 0);
            $sum += ($type === 'credit') ? $amt : -$amt;
        }
        return $sum;
    }

    private function walletLedger(int $userId): array
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('SELECT * FROM wallet_ledger WHERE user_id=? ORDER BY id DESC');
            $st->execute([$userId]);
            return $st->fetchAll(PDO::FETCH_ASSOC);
        }
        $all = $this->js->read('wallet_ledger');
        $rows = [];
        foreach ($all as $r) if ((int)($r['user_id'] ?? 0) === $userId) $rows[] = $r;
        usort($rows, fn($a,$b)=> (int)($b['id']??0) <=> (int)($a['id']??0));
        return $rows;
    }

    private function walletAdd(int $userId, string $type, int $amount, string $ref, string $meta=''): void
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('INSERT INTO wallet_ledger (user_id, type, amount, reference, meta, created_at) VALUES (?,?,?,?,?,?)');
            $st->execute([$userId, $type, $amount, $ref, $meta, now()]);
            return;
        }
        $all = $this->js->read('wallet_ledger');
        $all[] = [
            'id'=>$this->js->nextId($all),
            'user_id'=>$userId,
            'type'=>$type,
            'amount'=>$amount,
            'reference'=>$ref,
            'meta'=>$meta,
            'created_at'=>now(),
        ];
        $this->js->write('wallet_ledger', $all);
    }

    private function createPayment(int $userId, string $gateway, int $amount): int
    {
        $gateway = in_array($gateway, ['zibal','bitpay'], true) ? $gateway : 'unknown';
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('INSERT INTO payments (user_id,gateway,amount,status,authority,reference,meta,created_at) VALUES (?,?,?,?,?,?,?,?)');
            $st->execute([$userId,$gateway,$amount,'created',null,null,'{}',now()]);
            return (int)$pdo->lastInsertId();
        }
        $p = $this->js->read('payments');
        $id = $this->js->nextId($p);
        $p[] = ['id'=>$id,'user_id'=>$userId,'gateway'=>$gateway,'amount'=>$amount,'status'=>'created','authority'=>null,'reference'=>null,'meta'=>'{}','created_at'=>now()];
        $this->js->write('payments',$p);
        return $id;
    }

    private function updatePaymentStatus(int $paymentId, string $status, ?string $authority, ?string $reference, array $meta): void
    {
        if ($this->db->pdo()) {
            $pdo = $this->db->pdo();
            $st = $pdo->prepare('UPDATE payments SET status=?, authority=?, reference=?, meta=? WHERE id=?');
            $st->execute([$status, $authority, $reference, json_encode($meta, JSON_UNESCAPED_UNICODE), $paymentId]);
            return;
        }
        $p = $this->js->read('payments');
        foreach ($p as &$row) {
            if ((int)($row['id'] ?? 0) === $paymentId) {
                $row['status'] = $status;
                $row['authority'] = $authority;
                $row['reference'] = $reference;
                $row['meta'] = json_encode($meta, JSON_UNESCAPED_UNICODE);
                break;
            }
        }
        $this->js->write('payments', $p);
    }
}
