<?php

function h($v): string {
    return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8');
}

function ensure_dir(string $dir): void {
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
            throw new RuntimeException("Cannot create dir: $dir");
        }
    }
}

function request_method(): string {
    return strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
}

function request_uri_path(): string {
    $p = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
    return $p ?: '/';
}

function redirect(string $url): void {
    header('Location: ' . $url);
    exit;
}

function now(): string {
    return date('Y-m-d H:i:s');
}

function log_line(string $msg): void {
    $file = dirname(__DIR__) . '/logs/app.log';
    ensure_dir(dirname($file));
    file_put_contents($file, '['.date('c').'] '.$msg."\n", FILE_APPEND);
}

function app_config(): array {
    return $GLOBALS['APP_CONFIG'] ?? [];
}

function app_base_url(): string {
    $cfg = app_config();
    $base = (string)($cfg['app']['base_path'] ?? '');
    $base = rtrim($base, '/');
    return $base === '' ? '' : $base;
}

function app_url(string $path, array $query = []): string {
    $path = '/' . ltrim($path, '/');
    $url = app_base_url() . $path;
    if ($query) $url .= '?' . http_build_query($query);
    return $url;
}

function session_flash_set(string $msg): void {
    $_SESSION['_flash'] = $msg;
}

function session_flash_get(): ?string {
    $m = $_SESSION['_flash'] ?? null;
    unset($_SESSION['_flash']);
    return is_string($m) ? $m : null;
}

function csrf_token(string $key = '_csrf'): string {
    if (empty($_SESSION[$key]) || !is_string($_SESSION[$key])) {
        $_SESSION[$key] = bin2hex(random_bytes(32));
    }
    return $_SESSION[$key];
}

function csrf_field(string $key = '_csrf'): string {
    return '<input type="hidden" name="'.h($key).'" value="'.h(csrf_token($key)).'">';
}

function csrf_verify(string $key = '_csrf'): void {
    if (request_method() !== 'POST') return;
    $sent = $_POST[$key] ?? '';
    $sess = $_SESSION[$key] ?? '';
    if (!is_string($sent) || !is_string($sess) || !hash_equals($sess, $sent)) {
        http_response_code(419);
        exit('CSRF mismatch');
    }
}

function current_user(): ?array {
    $u = $_SESSION['user'] ?? null;
    return is_array($u) ? $u : null;
}

function require_login(): void {
    if (!current_user()) {
        session_flash_set('ابتدا وارد شوید.');
        redirect(app_url('/login'));
    }
}

function is_admin(): bool {
    $u = current_user();
    return $u && (($u['role'] ?? '') === 'admin');
}

function require_admin(): void {
    require_login();
    if (!is_admin()) {
        http_response_code(403);
        exit('Forbidden');
    }
}
