<?php
declare(strict_types=1);

$CONFIG = require dirname(__DIR__) . '/app/config.php';
$GLOBALS['APP_CONFIG'] = $CONFIG;

require dirname(__DIR__) . '/app/helpers.php';
require dirname(__DIR__) . '/app/Storage/Db.php';
require dirname(__DIR__) . '/app/Storage/JsonStore.php';
require dirname(__DIR__) . '/app/Models/Order.php';

date_default_timezone_set((string)($CONFIG['app']['timezone'] ?? 'Asia/Tehran'));

$db = new Db($CONFIG);
$js = null;
if (!$db->pdo()) {
    $dir = (string)($CONFIG['storage']['json_dir'] ?? dirname(__DIR__).'/data/json');
    $js = new JsonStore($dir);
}

$task = '';
if (PHP_SAPI === 'cli') {
    foreach ($argv as $arg) {
        if (str_starts_with($arg, 'task=')) $task = substr($arg, 5);
    }
} else {
    $task = (string)($_GET['task'] ?? '');
}

if ($task === '') {
    echo "Missing task\n";
    exit(1);
}

switch ($task) {
    case 'cleanup-otp':
        $removed = cleanupOtp($db, $js);
        echo "cleanup-otp removed=$removed\n";
        break;

    case 'advance-orders':
        $affected = advanceOrders($db, $js);
        echo "advance-orders affected=$affected\n";
        break;

    case 'reconcile-payments':
        // Placeholder: you can implement gateway verify loops here.
        echo "reconcile-payments TODO\n";
        break;

    default:
        echo "Unknown task\n";
        exit(1);
}

function cleanupOtp(Db $db, ?JsonStore $js): int
{
    $now = time();
    if ($db->pdo()) {
        $pdo = $db->pdo();
        $st = $pdo->prepare('DELETE FROM otp_codes WHERE expires_at < ?');
        $st->execute([$now]);
        return $st->rowCount();
    }
    $otp = $js->read('otp_codes');
    $before = count($otp);
    $otp = array_values(array_filter($otp, fn($r)=> (int)($r['expires_at'] ?? 0) >= $now));
    $js->write('otp_codes', $otp);
    return $before - count($otp);
}

function advanceOrders(Db $db, ?JsonStore $js): int
{
    $affected = 0;
    if ($db->pdo()) {
        $pdo = $db->pdo();
        $rows = $pdo->query("SELECT id, status FROM orders")->fetchAll(PDO::FETCH_ASSOC);
        foreach ($rows as $r) {
            $next = Order::statusFlow((string)$r['status']);
            if ($next !== $r['status']) {
                $pdo->prepare('UPDATE orders SET status=? WHERE id=?')->execute([$next, (int)$r['id']]);
                $affected++;
            }
        }
        return $affected;
    }
    $orders = $js->read('orders');
    foreach ($orders as &$o) {
        $next = Order::statusFlow((string)($o['status'] ?? 'pending'));
        if ($next !== ($o['status'] ?? '')) {
            $o['status'] = $next;
            $affected++;
        }
    }
    $js->write('orders', $orders);
    return $affected;
}
